#ifndef WIN32

#include "DesktopSharingControllerImplMac.h"
#include "csf/logger/CSFLogger.hpp"
#include "boost/bind.hpp"
#include "boost/function.hpp"

#include "jcfcoreutils/StringUtils.h"
#include "csfunified/services/interface/SystemService.h"
#include "csfunified/services/interface/ConfigService.h"
#include "csfunified/services/interface/ConfigValue.h"
#include "csfunified/services/interface/Presence.h"
#include "csfunified/services/interface/PresenceService.h"
#include "csfunified/services/interface/PresenceServiceCapabilities.h"
#include "csfsystemmonitor/SystemMonitor.h"
#include "csfsystemmonitor/SystemMonitorFactory.h"
#include "csfunified/framework/ServicesDispatcher.h"
#include "csfunified/framework/FunctorTask.h"
#include "jcfcoreutils/FunctionTask.h"
#include "../JabberServiceProvider.h"

const std::wstring WebEx_DS_Url = L"WebEx_DS_Url";
const std::wstring WebEx_DS_Token = L"WebEx_DS_Token";

namespace CSFUnified {
    
    static CSFLogger* DesktopSharingControllerImplLogger = CSFLogger_getLogger("MeetingService-DesktopSharingControllerImpl");
    
    DesktopSharingControllerImpl::DesktopSharingControllerImpl(SMART_PTR_NS::shared_ptr<UnifiedFactory> unifiedFactory)
    : DesktopSharingControllerBaseImpl(unifiedFactory)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::DesktopSharingControllerImpl()");
        
        m_unifiedFactory = unifiedFactory;
        m_started = false;
        m_strSessionName = "";
        
        m_pInstance = MeetingSDKWrap::getInstance();
        if (m_pInstance != NULL)
        {
            m_pDSMgr = m_pInstance->getDSMgr();
        }
    }
    
    DesktopSharingControllerImpl::~DesktopSharingControllerImpl()
    {
        m_strSessionName = "";
        m_started = false;
    }
    
    
    
    DesktopSharingControllerImpl * DesktopSharingControllerImpl::getInstance()
    {
        static DesktopSharingControllerImpl * m_dsController = NULL;
        if (NULL == m_dsController)
        {
            m_dsController = new DesktopSharingControllerImpl(JabberServiceProvider::getInstance().getUnifiedFactory());
        }
        
        return m_dsController;
    }
    
    // Feature set methods
    void DesktopSharingControllerImpl::registerDSCallback(SMART_PTR_NS::shared_ptr<DSCallback> dsCallback)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::registerDSCallback()");
        m_dsCallback = dsCallback.get();
    }
    
    void DesktopSharingControllerImpl::unregisterDSCallback(SMART_PTR_NS::shared_ptr<DSCallback> dsCallback)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::unregisterDSCallback()");
        m_dsCallback = NULL;
    }
    
    bool DesktopSharingControllerImpl::isInSharing()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::isInSharing()");
        if (m_pDSMgr != NULL)
        {
            return m_pDSMgr->isSharingInDS();
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::isInSharing(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::canStartSharing(std::string sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::canStartSharing()");
        if (m_started && m_pDSMgr != NULL)
        {
            return m_pDSMgr->canStartDS(sessionName.c_str());
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::canStartSharing(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::startSharing(std::string sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::startSharing()");
        if (m_pDSMgr != NULL)
        {
            CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::startSharing, session name = " << sessionName.c_str());
            m_strSessionName = sessionName;
            getDSToken();
            return true;
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::startSharing(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::canJoinSharing(std::string sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::canJoinSharing()");
        if (m_started && m_pDSMgr != NULL)
        {
            return m_pDSMgr->canJoinDS(sessionName.c_str());
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::canJoinSharing(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::joinSharing(std::string sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::joinSharing()");
        if (m_pDSMgr != NULL)
        {
            m_pDSMgr->joinDS(sessionName.c_str());
            return true;
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::joinSharing(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::stopSharing()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::stopSharing()");
        if (m_pDSMgr != NULL)
        {
            m_pDSMgr->endDS();
            return true;
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::stopSharing(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::reInviteCurSession()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::inviteUser2DS()");
        if (m_pDSMgr != NULL)
        {
            m_pDSMgr->inviteUser2DS(m_strSessionName.c_str());
            return true;
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::inviteUser2DS(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::restartSharing()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::restartSharing()");
        if (m_pDSMgr != NULL)
        {
            m_pDSMgr->reSharing();
            return true;
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::restartSharing(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::changePresenter()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::changePresenter()");
        if (m_pDSMgr != NULL)
        {
            m_pDSMgr->changePresenter();
            return true;
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::changePresenter(), m_pDSMgr is NULL");
        return false;
    }
    
    bool DesktopSharingControllerImpl::cancelSharing()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::cancelSharing()");
        return true;
    }
    
    bool DesktopSharingControllerImpl::declineSharing(std::string sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::declineSharing()");
        if (m_pDSMgr != NULL)
        {
            m_pDSMgr->declineDSInvitation(sessionName.c_str());
            return true;
        }
        
        CSFLogWarnS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::declineSharing(), m_pDSMgr is NULL");
        return false;
    }

    
    // IDSEvent
    void DesktopSharingControllerImpl::onPopupDSWaitingDialog()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onPopupDSWaitingDialog()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onShowStartingDialog(true);
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onShowStartingDialog, m_dsCallback, true), "DesktopSharingControllerImpl::onPopupDSWaitingDialog");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onCloseDSWaitingDialog()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onCloseDSWaitingDialog()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onShowStartingDialog(false);
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onShowStartingDialog, m_dsCallback, false), "DesktopSharingControllerImpl::onCloseDSWaitingDialog");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onDSStarted(const PString sessionName, bool isHost)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSStarted()");
        
        if (isHost)
        {
            reportDSUsage(L"START_MEETING");
        }
        else
        {
            reportDSUsage(L"ATTEND_MEETING");
        }
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onDSStarted(sessionName.c_str());
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onDSStarted, m_dsCallback, sessionName.c_str()), "DesktopSharingControllerImpl::onDSStarted");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onDSEnded(const PString sessionName, bool isHost)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSEnded()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onDSEnded(sessionName.c_str());
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onDSEnded, m_dsCallback, sessionName.c_str()), "DesktopSharingControllerImpl::onDSEnded");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onDSError(const DS_ERROR_CODE errorCode, const PString sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSError()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onDSError(sessionName.c_str(), errorCode);
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onDSError, m_dsCallback, sessionName.c_str(), errorCode), "DesktopSharingControllerImpl::onDSError");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onPopupChatWindow(const PString sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onPopupChatWindow()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onRequestRestoreWindow();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onRequestRestoreWindow, m_dsCallback), "DesktopSharingControllerImpl::onPopupChatWindow");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onMinimizeChatWindow(const PString sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onMinimizeChatWindow()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onRequestMinimizeWindow();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onRequestMinimizeWindow, m_dsCallback), "DesktopSharingControllerImpl::onMinimizeChatWindow");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onDSInvitation(const PString sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSInvitation()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onDSInvitationReceived(sessionName.c_str());
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onDSInvitationReceived, m_dsCallback, sessionName.c_str()), "DesktopSharingControllerImpl::onDSInvitation");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onDSDeclined(const PString sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSDeclined()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onDSDeclined(sessionName.c_str());
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onDSDeclined, m_dsCallback, sessionName.c_str()), "DesktopSharingControllerImpl::onDSDeclined");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onDSCancelled(const PString sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSCancelled()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onDSCancelled(sessionName.c_str());
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onDSCancelled, m_dsCallback, sessionName.c_str()),
                                                 "DesktopSharingControllerImpl::onDSCancelled");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onSharingStopped(const PString sessionName)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onSharingStopped()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onSharingStopped(sessionName.c_str());
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onSharingStopped, m_dsCallback, sessionName.c_str()), "DesktopSharingControllerImpl::onSharingStopped");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onBuddyJoined()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onBuddyJoined()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onUserJoined();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onUserJoined, m_dsCallback), "DesktopSharingControllerImpl::onBuddyJoined");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onUserTryLeave(bool isHost)
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onUserTryLeave()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onUserTryLeave(isHost);
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onUserTryLeave, m_dsCallback, isHost),
                                                 "DesktopSharingControllerImpl::onUserTryLeave");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onUserLeft()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onUserLeft()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onUserLeft();
            }
            else
            {
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onUserLeft, m_dsCallback), "DesktopSharingControllerImpl::onUserLeft");
            }
        }
    }
    
    void DesktopSharingControllerImpl::onPresenterChanged()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onPresenterChanged()");
        
        SMART_PTR_NS::shared_ptr<ServicesDispatcher> pServiceDispatcher = m_unifiedFactory->getServicesDispatcher();
        if (pServiceDispatcher != NULL && m_dsCallback != NULL)
        {
            bool bOnDispatcher = pServiceDispatcher->checkForUpdateAccess();
            if (bOnDispatcher)
            {
                m_dsCallback->onPresenterChanged();
            }
            else
            {                
                pServiceDispatcher->enqueueBlock(boost::bind(&DSCallback::onPresenterChanged, m_dsCallback), "DesktopSharingControllerImpl::onPresenterChanged");
            }
        }
    }
    
    //
    void DesktopSharingControllerImpl::start()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::start()");
        if (!m_started && m_pDSMgr != NULL)
        {
            m_pInstance->setIDSEvent(this);
            
            if (m_pInstance != NULL)
            {
                m_pInstance->startMeetingSDK();
            }
        }
        
        m_started = true;
    }
    
    void DesktopSharingControllerImpl::stop()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::stop()");
        if (m_started && m_pDSMgr != NULL)
        {
            if (m_pInstance != NULL)
            {
                m_pInstance->stopMeetingSDK();
            }
            
            m_pInstance->setIDSEvent(NULL);
        }
        
        m_started = false;
    }
    
    std::string DesktopSharingControllerImpl::getDSUrl()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::getDSUrl");
        
        std::string strDSUrl = "";
        if (m_unifiedFactory == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "m_unifiedFactory is NULL");
            return strDSUrl;
        }
        
        SMART_PTR_NS::shared_ptr<ConfigService> pConfigService = m_unifiedFactory->getService<ConfigService>();
        if (pConfigService == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get config service error");
            return strDSUrl;
        }
        
        SMART_PTR_NS::shared_ptr<ConfigValue> pDSUrlValue = pConfigService->getConfig(WebEx_DS_Url);
        if (pDSUrlValue == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get DS config value error");
            return strDSUrl;
        }
        
        strDSUrl = JCFCoreUtils::toString(pDSUrlValue->getValue());
        CSFLogDebugS(DesktopSharingControllerImplLogger, "get ds url = " << strDSUrl.c_str());
        
        return strDSUrl;
    }
    
    void DesktopSharingControllerImpl::getDSToken()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::getDSToken");
        
        if (m_unifiedFactory == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "m_unifiedFactory is NULL");
            return;
        }
        
        SMART_PTR_NS::shared_ptr<PresenceService> pPresenceService = m_unifiedFactory->getService<PresenceService>();
        if (pPresenceService == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get presence service error");
            return;
        }
        
        SMART_PTR_NS::shared_ptr<PresenceServiceNotifiers> pPresenceServiceNotifier = pPresenceService->getPresenceServiceNotifiers();
        if (pPresenceServiceNotifier == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get presence service notifier error");
            return;
        }
        
        SMART_PTR_NS::shared_ptr<PropertyNotifier> pDSTokenNotifier = pPresenceServiceNotifier->getDesktopShareTokenNotifier();
        if (pDSTokenNotifier == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get desktop share notifier error");
            return;
        }
        
        m_pDSTokenChanged = pDSTokenNotifier->connect(boost::bind(&DesktopSharingControllerImpl::onDSTokenValueChanged, this));
        pPresenceService->RenewDesktopShareToken();
    }
    
    void DesktopSharingControllerImpl::onDSTokenValueChanged()
    {
        CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSTokenValueChanged");
        
        if (m_pDSTokenChanged.lock() != NULL)
        {
            if (m_pDSTokenChanged.lock()->connected())
            {
                m_pDSTokenChanged.lock()->disconnect();
            }
        }
        
        SMART_PTR_NS::shared_ptr<PresenceService> pPresenceService = m_unifiedFactory->getService<PresenceService>();
        if (pPresenceService == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get presence service error");
            return;
        }
        std::string strDSToken = pPresenceService->getDesktopShareToken();
        CSFLogDebugS(DesktopSharingControllerImplLogger, "desktop share token" << strDSToken);
        
        std::string strDSUrl = getDSUrl();
        if (m_pDSMgr != NULL)
        {
            CSFLogDebugS(DesktopSharingControllerImplLogger, "DesktopSharingControllerImpl::onDSTokenValueChanged, session name = " << m_strSessionName.c_str() << "server url = " << strDSUrl << "DS token" << strDSToken.c_str());
            m_pDSMgr->startDS(m_strSessionName.c_str(), strDSUrl.c_str(), strDSToken.c_str());
        }
    }
    
    void DesktopSharingControllerImpl::reportDSUsage(std::wstring usageStr)
    {
        std::wstring eventName = usageStr;
        std::wstring extraData = L"<eventWorkspaceID></eventWorkspaceID>";
        std::wstring orgId = L"";
        std::wstring userId = L"";
        std::wstring groupId = L"";
        
        if (m_unifiedFactory == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "m_unifiedFactory is NULL");
            return;
        }
        
        SMART_PTR_NS::shared_ptr<ConfigService> pConfigService = m_unifiedFactory->getService<ConfigService>();
        if (pConfigService == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get config service error");
            return;
        }
        
        orgId = pConfigService->getConfig(L"WapiOrgId")->getValue();
        userId = pConfigService->getConfig(L"WapiUserId")->getValue();
        groupId = pConfigService->getConfig(L"WapiGroupId")->getValue();
        
        std::wostringstream extendedData;
        extendedData << L"<orgID>" << orgId << L"</orgID><userID>" << userId << L"</userID><groupID>" << groupId << L"</groupID>";
        
        SMART_PTR_NS::shared_ptr<SystemService> pSystemService = m_unifiedFactory->getService<SystemService>();
        if (pSystemService == NULL)
        {
            CSFLogWarnS(DesktopSharingControllerImplLogger, "get system service error");
            return;
        }
        CSFLogDebugS(DesktopSharingControllerImplLogger, "reportDSUsage eventName=" << eventName.c_str() << "extraData=" << extraData.c_str() << "extendedData=" << extendedData.str().c_str());
        pSystemService->SubmitUsageData(L"SYS_CUSTOM_EVENTS", eventName, extraData, extendedData.str());
    }
    

}

#endif